const express = require('express');
const cors = require('cors');
require('dotenv').config();

const { pool, testConnection } = require('./config/database');

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Root route - Database connection status
app.get('/', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Get database info using parameterized query
    const [dbInfo] = await connection.query('SELECT DATABASE() as db_name, VERSION() as db_version');
    
    connection.release();
    
    res.json({
      status: 'success',
      message: 'Database connected successfully',
      data: {
        connected: true,
        database: dbInfo[0].db_name,
        version: dbInfo[0].db_version,
        host: process.env.DB_HOST,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    res.status(500).json({
      status: 'error',
      message: 'Database connection failed',
      data: {
        connected: false,
        error: error.message,
        timestamp: new Date().toISOString()
      }
    });
  }
});

// Health check route
app.get('/health', (req, res) => {
  res.json({
    status: 'ok',
    timestamp: new Date().toISOString()
  });
});

// Sample API route to test database query
app.get('/api/test', async (req, res) => {
  try {
    const [rows] = await pool.query('SELECT 1 + 1 as result');
    res.json({
      status: 'success',
      data: rows[0]
    });
  } catch (error) {
    res.status(500).json({
      status: 'error',
      message: error.message
    });
  }
});

// Example CRUD endpoints with SAFE parameterized queries

// Get all users (with pagination)
app.get('/api/users', async (req, res) => {
  try {
    const limit = parseInt(req.query.limit) || 10;
    const offset = parseInt(req.query.offset) || 0;
    
    // Using parameterized queries to prevent SQL injection
    const [users] = await pool.query(
      'SELECT id, email, name, createdAt, updatedAt FROM users ORDER BY createdAt DESC LIMIT ? OFFSET ?',
      [limit, offset]
    );
    
    res.json({
      status: 'success',
      data: users
    });
  } catch (error) {
    res.status(500).json({
      status: 'error',
      message: error.message
    });
  }
});

// Create a new user (parameterized query)
app.post('/api/users', async (req, res) => {
  try {
    const { email, name } = req.body;
    
    // Validate input
    if (!email) {
      return res.status(400).json({
        status: 'error',
        message: 'Email is required'
      });
    }
    
    // Using parameterized query to prevent SQL injection
    const [result] = await pool.query(
      'INSERT INTO users (email, name, createdAt, updatedAt) VALUES (?, ?, NOW(), NOW())',
      [email, name || null]
    );
    
    // Fetch the created user
    const [users] = await pool.query(
      'SELECT id, email, name, createdAt, updatedAt FROM users WHERE id = ?',
      [result.insertId]
    );
    
    res.status(201).json({
      status: 'success',
      data: users[0]
    });
  } catch (error) {
    if (error.code === 'ER_DUP_ENTRY') {
      return res.status(400).json({
        status: 'error',
        message: 'Email already exists'
      });
    }
    res.status(500).json({
      status: 'error',
      message: error.message
    });
  }
});

// Get all posts (with pagination)
app.get('/api/posts', async (req, res) => {
  try {
    const limit = parseInt(req.query.limit) || 10;
    const offset = parseInt(req.query.offset) || 0;
    
    // Using parameterized query
    const [posts] = await pool.query(
      'SELECT id, title, content, published, authorId, createdAt, updatedAt FROM posts ORDER BY createdAt DESC LIMIT ? OFFSET ?',
      [limit, offset]
    );
    
    res.json({
      status: 'success',
      data: posts
    });
  } catch (error) {
    res.status(500).json({
      status: 'error',
      message: error.message
    });
  }
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({
    status: 'error',
    message: 'Route not found'
  });
});

// Start server
const startServer = async () => {
  // Test database connection before starting server
  await testConnection();
  
  app.listen(PORT, () => {
    console.log(`🚀 Server is running on port ${PORT}`);
    console.log(`📍 http://localhost:${PORT}`);
  });
};

startServer();

module.exports = app;
